#!/usr/bin/python3
from PIL import Image  # type: ignore
import typing

try:
    import colorama as colour  # type: ignore
except ImportError:
    _Fore_RED = ""
    _Fore_RESET = ""
else:
    _Fore_RED = colour.Fore.RED
    _Fore_RESET = colour.Fore.RESET

# These are integers which represent the suits of picture cards
CLUBS = 1
DIAMONDS = 2
HEARTS = 3
SPADES = 4

# This is a bit which gets turned on when the card is red (except picture cards)
REDCARD = 1 << 7

# All the following constants are based on windowed mode at 1600x900.
# Other sizes will not work (although it may be possible to rewrite
# the constants for those cases).

# This is how much border gets added to the window when the screenshot is taken
(_xoff, _yoff) = (3, 26)

# This is the top left of card (0,0) and the distance to the next card.
# The horizontal distance has been observed to be a non-integer (yuck).
(_cx, _cy, _cw, _ch) = (304, 382, 111.625, 25)

# This is the rectangle to be inspected relative to top-left of the card
(_nx, _ny, _nw, _nh) = (5, 6, 20, 10)

# Reference data for recognising the characters
_refdata = [
    [6, 3, 6, 2, -1, 6, 7, 7, 7, 5, 0],
    [7, 6, 6, 2, 2, 2, 2, 2, 2, 2, 1],
    [8, 3, 7, 7, 6, 5, 6, 7, 7, 6, 0],
    [9, 4, 7, 7, 7, 6, -1, 2, 6, 5, 0],
    [10, -1, 15, 15, 14, 14, 14, 14, 14, 15, 5],
    [CLUBS, 1, 3, 5, 3, 1, 10, 12, 12, 10, 0],
    [DIAMONDS, 0, 1, 3, 5, 7, 9, 11, 11, 9, 0],
    [HEARTS, 0, 0, 0, 9, 10, 12, 11, 9, 7, 0],
    [SPADES, 0, 2, 4, 6, 8, 10, 12, 11, 9, 0],
]

# This is how to find the image coordinates of the number on a card
def card_coords(col: int, row: int) -> typing.Tuple[int, int]:
    """Get the co-ordinates of a card on the page, given its row and column
    
    :param col: Column to get
    :type col: int
    :param row: Row to get
    :type row: int
    :return: Tuple of ints; the (x, y) coordinate of the card
    :rtype: Tuple[int, int]
    """
    return (_xoff + _cx + int(col * _cw + 0.5) + _nx, _yoff + _cy + _ch * row + _ny)


# recognise one card from image data given the row and column number
# of the card.  This does not currently allow the free cell to be recognised.
# The return value is a (possibly empty) list of candidates.
def recognise_one(
    im: Image, col: int, row: int, verbose: bool = False
) -> typing.List[int]:
    """Recognise one card from image data, given the row and column
    
    :param im: Screenshot of the field
    :type im: PIL.Image
    :param col: Column of the required card
    :type col: int
    :param row: Row of the required card
    :type row: int
    :param verbose: Whether or not to be verbose, defaults to False
    :type verbose: bool, optional
    :return: (possibly empty) list of candidates
    :rtype: List[int]
    """
    # Will be looking at RGB values so we need to know what format
    # the pixels will be in
    if im.getbands() != ("R", "G", "B"):
        print("invalid image format")
        return []
    (x, y) = card_coords(col, row)
    isred = None
    count = []
    # Stage 1: for each pixel row of the rectangle under consideration
    # we will find all the black (or red) pixels and record the distance
    # between the first and last (inclusive), or zero if there were none.
    for pixrow in range(_nh):
        min_value = -1
        max_value = 0
        frac = 0
        for pixcol in range(_nw):
            (r, g, b) = im.getpixel(  # pylint: disable=unused-variable
                (x + pixcol, y + pixrow)
            )
            # We can recognise red if we find a pixel where blue<<red
            if isred is None:
                if r > 160 and b < 88:
                    isred = True
                elif b < 64:
                    isred = False
                else:
                    continue
            # If the card is red, this formula applied to the blue value
            # will translate it to what the value would have been if the
            # card were black (obtained by inspection of similar black and
            # red images):
            if isred:
                b = int((b - 33) * 255 / 216)
            # Detect and save black pixels:
            if b - frac <= 48:
                frac = 0
                max_value = pixcol
                if min_value < 0:
                    min_value = pixcol
            # We have a scheme where two nearly-black pixels together can
            # count as a single black pixel because of anti-aliasing.  The
            # partial value is stored in 'frac'.
            elif b <= 96:
                frac = 96 - b
            else:
                frac = 0
        if min_value < 0:
            count.append(0)
        else:
            count.append(max_value - min_value + 1)

    # Stage 2: we will compare the list of counts to the reference data and
    # score each candidate on how closely it matches.  Keep a list of candidates
    # that scored the highest score, and also keep the second-highest score
    # as a measure of how certain the match is.
    # Note: -1 in the reference data means "don't care" and will match anything.
    max_value = 0
    last_max = 0
    candidates: typing.List[int] = []
    for ref in _refdata:
        card_type = ref[0]
        # We don't have to match the suit symbols if they are the wrong
        # colour.  Without this logic, diamonds look quite a lot like spades
        # because we are viewing them as shadows with the interior filled in.
        if (card_type == CLUBS or card_type == SPADES) and isred:
            continue
        if (card_type == DIAMONDS or card_type == HEARTS) and not isred:
            continue
        # In case this matches, record the red bit in the candidate's value
        if isred and card_type >= 6:
            card_type += REDCARD
        score = 0
        for i in range(10):
            n = ref[i + 1]
            if n == -1:
                score += 3
                continue
            d = abs(count[i] - n)
            if d <= 1:
                score += 3
            elif d == 2:
                score += 2
            elif d <= 4:
                score += 1
        if score > max_value:
            candidates = [card_type]
            last_max = max_value
            max_value = score
        elif score == max_value:
            candidates.append(card_type)
        elif score > last_max:
            last_max = score

    # Scoring is done: we should have the best score and a list of
    # matching candidates.
    # But if the score was bad, consider it a non-match.
    if max_value <= 15:
        candidates = []

    # End-game: report any wobbly matches and then return the answer.
    if verbose:
        intro = f"Card at ({col}, {row})"
        if len(candidates) == 0:
            print(intro + " not recognised")
        elif len(candidates) > 1:
            print(intro + " is non-unique")
        elif max_value - last_max < 5 and max_value < 30:
            print(intro + " is uncertain")

    return candidates


# Given a card type, returns a string to print out to represent that card
def display_card(card_type: int, use_colour: bool = True) -> str:
    """Return a given card's string representation
    
    :param card_type: The card to get, represented with its properties
    :type card_type: int
    :param use_colour: Whether or not to colour the output, defaults to True
    :type use_colour: bool, optional
    :return: The string used to represent the card
    :rtype: str
    """
    if use_colour:
        red = _Fore_RED
        reset = _Fore_RESET
    else:
        red = ""
        reset = ""
    if card_type == CLUBS:
        out_type = "C"
        red = ""
        reset = ""
    if card_type == DIAMONDS:
        out_type = "D"
    if card_type == HEARTS:
        out_type = "H"
    if card_type == SPADES:
        out_type = "S"
        red = ""
        reset = ""
    try:
        if card_type & REDCARD:
            out_type = str(card_type & ~REDCARD)
        else:
            red = ""
            reset = ""
    except:
        pass
    return f"{red}{out_type:>2}{reset} "


# Print out the board of cards
def printboard(cards: typing.List[typing.List[int]], use_colour: bool = True):
    """Print out the board of cards
    
    :param cards: 2D list of cards to be printed
    :type cards: List[List[int]]
    :param use_colour: Whether or not to style the output, defaults to True
    :type use_colour: bool, optional
    """
    row = 0
    ok = True
    while ok:
        output = ""
        ok = False
        for col in cards:
            if row < len(col):
                ok = True
                output += display_card(col[row], use_colour=use_colour)
            else:
                output += "  "
        if ok:
            print(output)
            row += 1


# recognise all cards from the given image data (assuming a 9x4 layout
# which is true when the cards were just dealt).
# Return a list of cards in each column.
def recognise(
    im: Image, verbose: bool = False, use_colour: bool = True
) -> typing.Optional[typing.List[typing.List[int]]]:
    """Recognise all cards from the given image data. Assumes 9x4 (true on fresh deal)
    
    :param im: Screenshot of game
    :type im: Image
    :param verbose: Whether or not to be verbose, defaults to False
    :type verbose: bool, optional
    :param use_colour: Whether or not to style output, if verbose, defaults to True
    :type use_colour: bool, optional
    :return: A 2D list of cards in each column
    :rtype: List[List[int]]
    """
    if verbose and use_colour:
        try:
            colour.init()
        except NameError:
            use_colour = False
            print("(No colour because colour wasn't imported)")
    if im.getbands() != ("R", "G", "B"):
        print("invalid image format")
        return None
    cards: typing.Union[
        typing.List[typing.List[int]], typing.List[typing.List[typing.Optional[int]]]
    ]
    cards = typing.cast(typing.List[typing.List[typing.Optional[int]]], [])
    fails: typing.List[typing.Tuple[int, int]] = []
    index: typing.Dict[int, int] = {}
    # Here is where we call the recogniser for each card on the board.
    # A fail is recorded if the result wasn't a single value.  A count
    # of the number of cards of each type is stored in "index".
    for col in range(9):
        column: typing.List[typing.Optional[int]] = []
        for row in range(4):
            candidates = recognise_one(im, col, row, verbose=verbose)
            if len(candidates) == 1:
                card = candidates[0]
                column.append(card)
                if card in index:
                    index[card] += 1
                else:
                    index[card] = 1
            else:
                column.append(None)
                fails.append((col, row))
        cards.append(column)

    # Validate the list of recognised cards using the index.
    # The layout should contain four of each suit and two of each
    # number in each of red and black.  If it contains more, then
    # the result is invalid.  If it contains less, the missing cards
    # are recorded.
    missing = []
    for card in [CLUBS, DIAMONDS, HEARTS, SPADES]:
        if card not in index:
            i = 0
        else:
            i = index[card]
        if i > 4:
            if verbose:
                print(
                    f"Error: {i} of card",
                    display_card(card, use_colour=use_colour),
                    "found",
                )
            return None
        while i < 4:
            missing.append(card)
            i += 1
    for num in range(6, 10):
        for card in [num, num + REDCARD]:
            if card not in index:
                i = 0
            else:
                i = index[card]
            if i > 2:
                if verbose:
                    print(
                        f"Error: {i} of card",
                        display_card(card, use_colour=use_colour),
                        "found",
                    )
                return None
            while i < 2:
                missing.append(card)
                i += 1
    # The number of missing cards recorded should be the same as the number
    # of failed recognitions.
    if len(missing) != len(fails):
        if verbose:
            print("Error: wrong missing card count")
        return None
    # If there was one failure and one missing card, we know that the missing
    # card  should be in the failed slot.
    if len(missing) == 1:
        if verbose:
            print("Interpolating 1 missing card")
        (x, y) = fails[0]
        cards[x][y] = missing[0]
    # There could be a whole bunch of logic here to try to work out what the
    # missing cards are if there were two or more failures. But there isn't.
    elif len(missing) > 1:
        if verbose:
            print(
                "Ideally we would have logic here to reconcile the "
                + str(len(missing))
                + " cards, but we didn't write it"
            )
        return None
    else:
        if verbose:
            print("Recognition all checks out")
    cards = typing.cast(typing.List[typing.List[int]], cards)
    # Finally: print out the board and return it.
    if verbose:
        printboard(cards, use_colour=use_colour)

    return cards
